const $ = (sel) => document.querySelector(sel);
const $$ = (sel) => Array.from(document.querySelectorAll(sel));

const formatterCOP = new Intl.NumberFormat("es-CO", { style: "currency", currency: "COP", maximumFractionDigits: 0 });
const formatterPct = new Intl.NumberFormat("es-CO", { style: "percent", maximumFractionDigits: 2 });
const formatterNum = new Intl.NumberFormat("es-CO", { maximumFractionDigits: 2 });

const state = {
  costoTotal: 3000000,
  pesoTotalRef: 80,
  filas: []
};

function nuevaFila(prefill = {}) {
  const f = { pieza: prefill.pieza ?? "", peso: prefill.peso ?? 0, precio: prefill.precio ?? 0 };
  state.filas.push(f);
  render();
}
function eliminarFila(idx) { state.filas.splice(idx, 1); render(); }

function calcular() {
  const ingresos = state.filas.map(f => (Number(f.peso)||0) * (Number(f.precio)||0));
  const sumaIngresos = ingresos.reduce((a,b)=>a+b,0);
  const proporciones = ingresos.map(x => sumaIngresos>0 ? x/sumaIngresos : 0);
  const costosAsignados = proporciones.map(p => p * state.costoTotal);
  const costoKg = state.filas.map((f,i)=> {
    const peso = Number(f.peso)||0;
    return peso>0 ? costosAsignados[i]/peso : 0;
  });
  const utilidades = ingresos.map((ing,i)=> ing - costosAsignados[i]);
  const margenes = utilidades.map((u,i)=> costosAsignados[i]>0 ? u/costosAsignados[i] : 0);
  const totalPeso = state.filas.reduce((a,f)=>a+(Number(f.peso)||0),0);
  const totalCosto = costosAsignados.reduce((a,b)=>a+b,0);
  const totalUtilidad = utilidades.reduce((a,b)=>a+b,0);
  const sumaProporciones = proporciones.reduce((a,b)=>a+b,0);
  return { ingresos, sumaIngresos, proporciones, costosAsignados, costoKg, utilidades, margenes, totalPeso, totalCosto, totalUtilidad, sumaProporciones };
}

function render() {
  state.costoTotal = Number($("#costoTotal").value) || 0;
  state.pesoTotalRef = Number($("#pesoTotal").value) || 0;

  const calc = calcular();
  const tbody = $("#tbody");
  tbody.innerHTML = "";

  state.filas.forEach((fila, idx) => {
    const tr = document.createElement("tr");
    tr.className = "border-t hover:bg-slate-50";
    tr.innerHTML = `
      <td class="p-3"><input class="w-full px-2 py-1 rounded-lg border" value="${fila.pieza}" data-k="pieza" data-i="${idx}" /></td>
      <td class="p-3 text-right"><input type="number" step="0.01" class="number-input w-28 px-2 py-1 rounded-lg border text-right" value="${fila.peso}" data-k="peso" data-i="${idx}" /></td>
      <td class="p-3 text-right"><input type="number" step="1" class="number-input w-28 px-2 py-1 rounded-lg border text-right" value="${fila.precio}" data-k="precio" data-i="${idx}" /></td>
      <td class="p-3 text-right">${formatterCOP.format(calc.ingresos[idx]||0)}</td>
      <td class="p-3 text-right">${formatterPct.format(calc.proporciones[idx]||0)}</td>
      <td class="p-3 text-right">${formatterCOP.format(calc.costosAsignados[idx]||0)}</td>
      <td class="p-3 text-right">${formatterCOP.format(calc.costoKg[idx]||0)}</td>
      <td class="p-3 text-right ${(calc.utilidades[idx]||0)<0?'text-red-600':'text-emerald-700'}">${formatterCOP.format(calc.utilidades[idx]||0)}</td>
      <td class="p-3 text-right">${formatterPct.format(calc.margenes[idx]||0)}</td>
      <td class="p-3 text-center"><button class="px-2 py-1 rounded-lg bg-rose-600 text-white hover:bg-rose-700" data-del="${idx}">Eliminar</button></td>
    `;
    tbody.appendChild(tr);
  });

  $("#tPeso").textContent = formatterNum.format(calc.totalPeso);
  $("#tIngreso").textContent = formatterCOP.format(calc.sumaIngresos||0);
  $("#tCosto").textContent = formatterCOP.format(calc.totalCosto||0);
  $("#tUtilidad").textContent = formatterCOP.format(calc.totalUtilidad||0);
  $("#tProp").textContent = "—";

  $("#chkProp").textContent = (calc.sumaProporciones||0).toFixed(2);
  $("#chkCosto").textContent = formatterCOP.format(calc.totalCosto||0);
  $("#chkDiff").textContent = formatterCOP.format((calc.totalCosto||0) - (state.costoTotal||0));

  $$("#tbody input").forEach(inp => {
    inp.addEventListener("input", (e)=>{
      const i = Number(e.target.dataset.i);
      const k = e.target.dataset.k;
      let val = e.target.value;
      if(k!=="pieza") val = Number(val)||0;
      state.filas[i][k] = val;
      render();
    });
  });
  $$("#tbody [data-del]").forEach(btn => btn.addEventListener("click", (e)=> eliminarFila(Number(e.target.dataset.del))));
}

$("#costoTotal").addEventListener("input", ()=>render());
$("#pesoTotal").addEventListener("input", ()=>render());

$("#btnAgregar").addEventListener("click", ()=> nuevaFila({ pieza:"", peso:0, precio:0 }));
$("#btnNuevo").addEventListener("click", ()=> { state.filas = []; render(); });
$("#btnEjemplo").addEventListener("click", ()=> {
  state.filas = [
    { pieza:"Lomo fino", peso:8, precio:55000 },
    { pieza:"Costilla", peso:10, precio:42000 },
    { pieza:"Pierna", peso:12, precio:45000 },
    { pieza:"Pecho", peso:6, precio:38000 },
    { pieza:"Cadera", peso:10, precio:47000 },
    { pieza:"Pulpa", peso:8, precio:50000 },
    { pieza:"Hueso y grasa", peso:6, precio:8000 },
    { pieza:"Menudencias", peso:4, precio:20000 }
  ];
  render();
});

$("#btnGuardar").addEventListener("click", ()=>{
  const data = JSON.stringify(state);
  const blob = new Blob([data], { type:"application/json" });
  const a = document.createElement("a");
  a.href = URL.createObjectURL(blob);
  a.download = "costeador-res.json";
  a.click();
  URL.revokeObjectURL(a.href);
});

$("#btnCargar").addEventListener("click", ()=>{
  const inp = document.createElement("input");
  inp.type = "file"; inp.accept = "application/json";
  inp.onchange = (e)=>{
    const file = e.target.files[0];
    const reader = new FileReader();
    reader.onload = ()=>{
      try{
        const loaded = JSON.parse(reader.result);
        state.costoTotal = Number(loaded.costoTotal) || state.costoTotal;
        state.pesoTotalRef = Number(loaded.pesoTotalRef) || state.pesoTotalRef;
        state.filas = Array.isArray(loaded.filas) ? loaded.filas : [];
        $("#costoTotal").value = state.costoTotal;
        $("#pesoTotal").value = state.pesoTotalRef;
        render();
      }catch(_){ alert("Archivo inválido."); }
    };
    reader.readAsText(file);
  };
  inp.click();
});

$("#btnExportar").addEventListener("click", ()=>{
  const rows = [["Pieza","Peso (kg)","Venta/kg (COP)","Ingreso","% Costo","Costo asignado","Costo/kg","Utilidad","Margen %"]];
  const c = (n)=>Math.round(n||0);
  const calc = calcular();
  state.filas.forEach((f,i)=>{
    rows.push([f.pieza, f.peso, f.precio, c(calc.ingresos[i]), calc.proporciones[i], c(calc.costosAsignados[i]), c(calc.costoKg[i]), c(calc.utilidades[i]), calc.margenes[i]]);
  });
  rows.push(["TOTAL", calc.totalPeso, "", c(calc.sumaIngresos), calc.sumaProporciones, c(calc.totalCosto), "", c(calc.totalUtilidad), ""]);
  const csv = rows.map(r => r.map(x => typeof x==="string" ? `"${x.replace(/"/g,'""')}"` : x).join(",")).join("
");
  const blob = new Blob([csv], { type: "text/csv;charset=utf-8" });
  const a = document.createElement("a");
  a.href = URL.createObjectURL(blob);
  a.download = "costeador-res.csv";
  a.click();
  URL.revokeObjectURL(a.href);
});

render();
